#ident "$Id: aspi.c,v 1.1 1992/01/15 01:14:12 chris Exp $"

#include <stdlib.h>
#include <io.h>
#include <stdio.h>
#include <fcntl.h>
#include <dos.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include <errno.h>

/*
 * $Log: aspi.c,v $
 * Revision 1.1  1992/01/15  01:14:12  chris
 * Initial revision
 *
 *
 *
 * chris@alderan.sdata.de
 */

#define DWORD unsigned long
#ifndef TAPE_ID
#define TAPE_ID "0:4:0"
#endif

#include "aspi.h"
#include "scsi.h"
#include "scsierr.h"

#ifndef CTCTRL
#include "tar.h"
#endif

static char *aspi_c_id = "$Id: aspi.c,v 1.1 1992/01/15 01:14:12 chris Exp $";

void aspiinit(void);
int  aspiclose(int);
static int  setRecsize(void);

/*
 * The complete scsi address of our tape device
 */
int adapter_id, scsi_tape_id, lun_id;

/*
 * Because all the scsi functions start with:
 * SCSIfunc(adapter_id, scsi_tape_id, lun, .......)
 * .. we define a macro "TARGET", so that we can write SCSIfunc(TARGET, ...);
 */
#define TARGET adapter_id, scsi_tape_id, lun_id

#ifndef CTCTRL

/* Flags to for use with the streamer (detected at aspiopen()) */
static int aspifile, no_rewind;

/* This holds the max., and min. recordsize, the streamer can deal with */
static blocklimit lim;

static unsigned long    physRecSize = 0L;   /* SCSI blocksize */
static char             denCode = 0x0;      /* SCSI density code */

static long apos;       /* this holds the current archive file pointer */
static int write_file_mark_on_close;
static int end_of_medium;
static unsigned char rsense[14]; /* array for scsi-sense-data */

/*
 * "aspiPhysRec()" and "aspiDenCode()" are used to pass the SCSI
 *   blocksize and density codes into this module.
 */
void
aspiPhysRec(
    int recsize
    )
{
    physRecSize = (unsigned long)recsize;
}

void
aspiDenCode(
    int density
    )
{
    denCode = (char)density;
}

/*
 * The next two functions are called when you open a file.
 * They check if the filename specifies the scsi-tape.
 * The filenames used for that are "/dev/ct" - "rewind cartridge tape"
 * and "/dev/nrct" - "no rewind cartridge tape" (no rewind on close)
 * I just choose those names "/dev/ct" and "/dev/nrct" because the
 * filenames are unusual to exist on a MSDOS filesystem and also because
 * I'm used to them from my ix386. Anyway, there are no other dependencies
 * on those names in the whole program. So, if you prefer to change the
 * names to "/dev/rmt0" or whatever, just change them !
 * As you can see in aspicreate() and aspiopen(), I still open a
 * MSDOS-output-dummy-file. "nul:" is kind of /dev/null for MSDOS
 * and I had to do this because the MSDOS code from tar still
 * does some operations on this file like setmode(stream) etc. and
 * I was too lazy to change the code in tar. So, with this they
 * may do setmode()'s on the null-device as long as they want :-)
 */


/*
 * Create an archive file
 */

aspicreat(path, mode)
char *path;
int mode;
{
  end_of_medium = 0;

  if ( !strcmp(path, "/dev/ct") )  {
    apos = 0l;
    no_rewind = 0;
    aspiinit();
    write_file_mark_on_close = 1;
    return (aspifile=creat("NUL", mode));
  } else if ( !strcmp(path,"/dev/nrct") )  {
    apos = 0l;
    no_rewind = 1;
    aspiinit();
    write_file_mark_on_close = 1;
    return (aspifile=creat("NUL", mode));
  }
  aspifile = -1;
  return creat(path, mode);
}

/*
 * Open an archive file
 */

aspiopen(path, oflag, mode)
char *path;
int oflag;
int mode;
{
  end_of_medium = 0;

  if ( !strcmp(path, "/dev/ct") )  {
    apos = 0l;
    no_rewind = 0;
    aspiinit();
    if ( (oflag & O_WRONLY) || (oflag & O_RDWR) )
      write_file_mark_on_close = 1;
    else write_file_mark_on_close = 0;
    return (aspifile=open("NUL", oflag, mode));
  } else if ( !strcmp(path,"/dev/nrct") )  {
    apos = 0l;
    no_rewind = 1;
    aspiinit();
    if ( (oflag & O_WRONLY) || (oflag & O_RDWR) )
      write_file_mark_on_close = 1;
    else write_file_mark_on_close = 0;
    return (aspifile=open("NUL", oflag, mode));
  }
  aspifile = -1;
  return open(path, oflag, mode);
}


/*
 * So, all the other functions now just check if (fileds == aspifile)
 * and do redirect the operation to the tape. Otherwise they just
 * hand the request to the original function (e.g aspiread()->read() etc.).
 */


/*
 * Read data from an archive file.  (NOTE:  If fixed-block mode is
 *  used, then the buffer MUST be an even multiple of the physical
 *  record size.)
 */

aspiread(
    int fileds,
    char *buf,
    int nbyte
    )
{
  int       i;
  unsigned  fixed;
  DWORD     xferlen;

  if ( fileds == aspifile )  {

    /*
     * Determine fixed or variable block mode and set transfer
     *  length accordingly.
     */
    fixed = (physRecSize==0) ? 0x0 : 0x1;
    if (fixed) {
        if ((unsigned)nbyte % physRecSize)  {
            fprintf(stderr,
                    "aspiread: Illegal blocklen for tape: %u\n", nbyte);
            aspiclose(fileds);
            exit(EX_SYSTEM);
        }
        xferlen = (DWORD)((unsigned)nbyte / physRecSize);
    } else {
        xferlen = (DWORD)(unsigned)nbyte;
    }

    if (i = SCSIRead(TARGET, buf, xferlen, 0, fixed, rsense)) {
        unsigned bytes_read;

        if ( (i == E$BlankCheck)
          || ((i == E$Medium) && (rsense[2]&0x40) && (rsense[0]&0x80)) ) {

            /* End-of-data or End-of-partition */
            errno = ENOSPC;
            if ( (rsense[2] & 0x40) && (rsense[0] &0x80) ) {
                /* compute the number of bytes read */
                long    info;
                info = ((DWORD)rsense[3]<<24) |
                       ((DWORD)rsense[4]<<16) |
                       ((DWORD)rsense[5]<<8)  |
                       ((DWORD)rsense[6]);
                bytes_read = (unsigned)nbyte
                           - (unsigned)(info * (fixed ? physRecSize : 1));
                apos += bytes_read;
                return (int)bytes_read;
            } else {
                /* Residue not valid, fatal error */
                aspiclose(fileds);
                exit(EX_SYSTEM);
            }
        } else  {
            /* Some other error - fatal */
            fprintf(stderr,"aspiread: errno: %d\n", i);
            aspiclose(fileds);
            exit(EX_SYSTEM);
        }
    }
    apos += (unsigned)nbyte;
    return(nbyte);
  }
  return(read(fileds, buf, nbyte));
}

/*
 * Write data to an archive file
 */

aspiwrite(
    int     fileds,
    char    *buf,
    int     nbyte
    )
{
  int       i;
  unsigned  fixed;
  DWORD     xferlen;


  if ( fileds == aspifile )  {

    /*
     * Since WRITE's past "Early Warning" continue to accept data
     *  and the only way to let the caller know that an error has
     *  occurred is to write less data that was requested, we need
     *  to remember that the previous WRITE hit "Early Warning" and
     *  stop accepting data.
     */
    if (end_of_medium) {
        errno = ENOSPC;
        return(0);
    }

    /*
     * Determine fixed or variable block mode and set transfer
     *  length accordingly.
     */
    fixed = (physRecSize==0) ? 0x0 : 0x1;
    if (fixed) {
        if ((unsigned)nbyte % physRecSize)  {
            fprintf(stderr,
                    "aspiwrite: Illegal blocklen for tape: %u\n", nbyte);
            aspiclose(fileds);
            exit(EX_SYSTEM);
        }
        xferlen = (DWORD)((unsigned)nbyte / physRecSize);
    } else {
        xferlen = (DWORD)(unsigned)nbyte;
    }

    if (i = SCSIWrite(TARGET, buf, xferlen, fixed, rsense)) {
        unsigned    bytes_written;

        if ( (i == E$EndOfMedium) && (rsense[0] & 0x80) ) {
            /*
             * Early warning encountered.  Need to remember this
             *  to trip the special-case error reporting above.  (The
             *  things we do to keep compatible with "write()". - sigh)
             */
            long    info;
            errno = ENOSPC;
            end_of_medium = 1;

            info = ((DWORD)rsense[3]<<24) |
                   ((DWORD)rsense[4]<<16) |
                   ((DWORD)rsense[5]<<8)  |
                   ((DWORD)rsense[6]);
            bytes_written = (unsigned)nbyte
                          - (unsigned)(info * (fixed ? physRecSize : 1));
            apos += bytes_written;

            return((int)bytes_written);
        }
        /*
         * Any error other that early warning with a valid residue
         *  is fatal.
         */
        fprintf(stderr,"aspiwrite: errno: %d\n", i);
        aspiclose(fileds);
        exit(EX_SYSTEM);
    }
    /* No error occurred */
    apos += (unsigned)nbyte;
    return(nbyte);
  }
  return write(fileds, buf, nbyte);
}

/*
 * close an archive file
 */

aspiclose(fileds)
int fileds;
{
  int i;

  if ( fileds == aspifile )  {
    /*
     * First, write a filemark!  There should be room for this
     *  even if "Early Warning" has been hit.
     */
    if ( write_file_mark_on_close ) {
      if ( (i=SCSIWriteFilemarks(TARGET, 1l, 0, 0, rsense)) ) {
        /*
         * Not really an error if just "Early Warning" with no residue.
         */
        if ( (E$EndOfMedium != i) || !(rsense[0] & 0x80)
          || rsense[3] || rsense[4] || rsense[5] || rsense[6] ) {
            fprintf(stderr,"aspiclose: Error writing filemark\n");
        }
      }
    }
    /* then rewind the tape if we have to */
    if ( ! no_rewind )  {
      if ( (i=SCSIRewind(TARGET, 0, 0)) )
        ( void ) fprintf(stderr,"aspiclose: Error rewinding the tape\n");
    } else if ( ! write_file_mark_on_close )  {
     /*
      * This means we've been reading an archive on a "/dev/nrct" tape.
      * In this case we have seek over the filemark at the end of the
      * archive.
      */
      if ( (i=SCSISpace(TARGET, 1, 1l, 0))  )
        ( void ) fprintf(stderr,"aspiclose: Error seeking filemark\n");
    }
    apos = 0l;
  }
  return close(fileds);
}

/*
 * Seek on the archive file
 */

long
aspilseek(
    int     fileds,
    long    offset,
    int     whence
    )
{
  int i;
  long newpos;

  if ( fileds == aspifile )  {
   /*
    * NOTE: seeking backwards is not supported by every streamer.
    * The Archive Viper 2150S does, but I don't know about others.
    */
    switch ( whence )  {
    case SEEK_SET:
      newpos = offset;
      break;
    case SEEK_CUR:
      newpos = apos + offset;
      break;
    default:
      fprintf(stderr, "aspilseek: mode %d not supported\n", whence);
      aspiclose(fileds);
      exit(EX_SYSTEM);
      break;
    }
    if ( newpos % blocksize )  {
      fprintf(stderr,
           "aspilseek: can't seek to a non-block-boundary position (%ld)\n",
           newpos);
      aspiclose(fileds);
      exit(EX_SYSTEM);
    }
    if ( (i=SCSISpace(TARGET, 0x0,
            (newpos-apos)/((physRecSize==0)?(long)blocksize:physRecSize),
            0)) ) {

      fprintf(stderr, "aspilseek: SCSISpace retuned errno %d\n", i);
      aspiclose(fileds);
      exit(EX_SYSTEM);
    }
    apos = newpos;
    return newpos;
  }
  return lseek(fileds, offset, whence);
}

#endif  /* CTCTRL */

/*
 * Here we start with all that aspi stuff !!!
 * --------------------------------------------
 * the way we get the entrypoint of the aspi - module is quite strange,
 * but anyway ..
 */

/*
 * This will be the function for all aspi requests, it gets
 * initialized at aspiinit() and called from aspirequest()
 */

void (far *aspi)(void far *) = (void far *) 0;


/*
 * Initialize aspi and the tape device
 */

void aspiinit()
{
  int h,i;
  union REGS inregs;
  union REGS outregs;
  char *getenv(), *tapeid;

  if ( (h=open("SCSIMGR$",0)) == -1 )  {
    perror("Opening ASPI Manager");
    exit(1);
  }

/*
 * This is an ioctl(READ) to the "SCSIMGR$".
 * It returns the entrypoint of the aspi-module (far pointer)
 * in the 4 bytes, pointed to by the dx register.
 */

  inregs.x.ax = 0x4402;
  inregs.x.dx = ( int ) &aspi;
  inregs.x.cx = 4;
  inregs.x.bx = h;
  intdos(&inregs, &outregs);

 /*
  * Now find out on which device we have to operate
  */
  tapeid = getenv("TAPEID");
  if ( !tapeid ) tapeid = TAPE_ID;
  if ( (tapeid[1] != ':') || (tapeid[3] != ':') ||
    (!isdigit(tapeid[0])) || (!isdigit(tapeid[2])) || (!isdigit(tapeid[4])) ) {
    ( void ) fprintf(stderr,"aspiinit: Illegal TAPEID: \"%s\"\n", tapeid);
    exit(1);
  } else  {
    adapter_id = tapeid[0] - '0';
    scsi_tape_id = tapeid[2] - '0';
    lun_id = tapeid[4] - '0';
  }

#ifndef CTCTRL

/*
 * Now, check the device ...
 */
  if ( (i=GetDeviceTyp( TARGET )) < 0 )  {
    ( void ) fprintf(stderr,
             "aspiinit: Can't determine type of device \"%s\"\n", tapeid);
    ( void ) fprintf(stderr,"aspiinit: GetDeviceTyp() returned errno %d\n", i);
    exit(EX_SYSTEM);
  } else if ( i != 1 )  {
    ( void ) fprintf(stderr,"aspiinit: device \"%s\" is not a streamer !\n",
                                                                   tapeid);
    ( void ) fprintf(stderr,"the type returned by GetDeviceTyp was %d\n",i);
    exit(EX_SYSTEM);
  }

/*
 * Select an appropriate physical record size and recording density
 *  and initialize the device to the selected values. - R. Rose (Feb '93)
 */
  if (setRecsize()!=0) {
    fprintf(stderr,"aspiinit: Can't set blocksize/density for device\n");
    exit(EX_SYSTEM);
  }

#endif /* CTCTRL */

}

#ifndef CTCTRL
/*
 * Set the physical record size and recording density for the
 *  SCSI device to something reasonable or whatever was requested
 *  on the tar command line. - R. Rose (Feb '93)
 */
static int
setRecsize()
{
    int     rc;
    int     retries;
    modeBlockSz modeBlockData;

    /*
     * Get the max. and min. recordsize for the streamer.  If the
     *  drive returns a "Unit Attention" condition, retry the operation
     *  a few times.  Normally, "Unit Attention" will clear out in 1-3
     *  tries.  (Actually, we ought to forget this retry loop and use
     *  a "Test Unit Ready" to clear out the "Unit Attention" and wait
     *  a reasonable time for the drive to stop returning "Busy" and
     *  "Not Ready".  Maybe if I get time, I'll come back and do that.)
     */
    retries = 5;
    while ( E$NoErr != (rc=SCSIReadBlockLimits(TARGET, &lim, 0)) ) {

        switch (rc) {

        case E$UnitAttention:
            if (retries-- > 0) {
                break;
            }
            /* fallthru */
        default:
            fprintf(stderr,
              "setRecsize: Can't read blocklimits. errno %d\n", rc);
            return(-1);
        }
    }

    /*
     * Get the current physical record size and recording density for the
     *  SCSI device.
     */
    if (E$NoErr != (rc=SCSIModeSense(TARGET, &modeBlockData, 0))) {
        /*
         * Mode Sense failed, so we punt.  If the data from the
         *  Read Block Limits showed the device to be fixed-block
         *  with a blocksize that evenly divides into the tar blocking
         *  factor and the density-code specifies "default" (0x0) or
         *  "no change" (0x7F), then use the device's native blocksize.
         */
        if ( ((denCode!=0x0) && (denCode!=0x7F))
          || (lim.min != lim.max) || (blocksize % lim.min) ) {
            fprintf(stderr,
              "setRecsize: Can't get blocksize/density. errno %d\n", rc);
            return(-1);
        }
        physRecSize = lim.min;
        return(0);
    }
    if ( ((unsigned)modeBlockData.header.dataLen < 11)
      || ((unsigned)modeBlockData.header.bdescLen != 8) ) {
        /*
         * Drive doesn't support all of the Mode Sense data we need.
         *  Fatal error, since this should have worked.
         */
        fprintf(stderr,
          "setRecsize: Incomplete blocksize/density data from device\n");
        return(-1);
    }
    if (lim.min == lim.max) {
        /*
         * Fixed block device, so SCSI blocksize must be variable or
         *  match the device's blocksize.  Also, the device's blocksize
         *  must evenly must evenly divide into the tar blocking factor.
         */
        if ( ((physRecSize!=0) && (physRecSize!=lim.min))
          || (blocksize % lim.min) ) {
            fprintf(stderr,
              "setRecsize: Blocking/physical blocksize mismatch\n");
            fprintf(stderr,
              "          fixed-block device length = %lu\n",lim.min);
            return(-1);
        }
        physRecSize = lim.min;
    } else {
        /*
         * Device supports both fixed and variable-block mode transfers.
         *  If variable-block mode is used, then the tar blocking factor
         *  must be within the block limit range.  If fixed-mode is used,
         *  then the specified blocksize must evenly map into the tar
         *  blocking factor.  (Don't worry about validating the physical
         *  record size for fixed block, since it should fail on the
         *  Mode Select if it's out of range.)
         */
        if (physRecSize == 0L) {
            if ( ((unsigned long)blocksize<lim.min)
              || ((lim.max != 0L) && ((unsigned long)blocksize>lim.max)) ) {
                fprintf(stderr,
                  "setRecsize: Tar blocksize out of physical range.\n");
                fprintf(stderr,
                  "          Device minimum = %lu, maximum = %lu\n",
                  lim.min, lim.max);
                return(-1);
            }
        } else {
            if (blocksize % physRecSize) {
                fprintf(stderr,
                  "setRecsize: Blocking/physical blocksize mismatch.\n");
                return(-1);
            }
        }
    }

    /*
     * Issue Mode Select wth PF bit of 1 to change the density and
     *  blocksize.  If this fails with an "Illegal Request", then try a
     *  Mode Select with a PF bit of 0.
     */
    modeBlockData.header.dataLen = 0x0;        /* Reserved on Mode Select */
    modeBlockData.header.mediumType = 0x0;     /* Reserved on Mode Select */
    modeBlockData.header.devSpecific &= ~0x80; /* Clear Write Protect bit */
    modeBlockData.bdesc.denCode = denCode;     /* Recording density       */
    modeBlockData.bdesc.nBlocks = 0L; /* Set blocksize for remaining tape */
    modeBlockData.bdesc.blockLen = physRecSize;

    if (E$NoErr == (rc=SCSIModeSelect(TARGET,1,&modeBlockData,0))) {
        return(0);
    }
    if ( (rc == E$IllegalReq)
      && (E$NoErr == (rc=SCSIModeSelect(TARGET,0,&modeBlockData,0))) ) {
        return(0);
    }
    fprintf(stderr,"setRecsize:  Mode Select failed.  aspiErr=%d\n",rc);
    return(-1);
}
#endif /* CTCTRL */

/*
 * This is the function that we call for all out aspi-requests.
 * The function simply takes our requestblock and hands it to
 * aspi() (the real aspi request function) and then it polls
 * and waits for the request to finish.
 */

int aspireq(srb)
void *srb;
{
  int cnt;

  struct _srbinquiry *s;
  s = ( struct _srbinquiry *) srb;
  aspi(s);
  cnt = 10;
  while ( cnt-- ) while ( ! s->h.status )  ;   /* POLL ! */
  if ( s->h.status == 1 ) return E$NoErr;
  return E$AspiErr;
}

/*
 * The following two commandos are supported directly by the ASPI module
 * and do not require the construction of a SCSI-CCB.
 */

int HostAdapterInquiry(inq)
aspiinquiry *inq;
{
  struct _srbinquiry *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) in the lower 1 MB */

  if ( (srb=calloc(sizeof(struct _srbinquiry), 1)) == NULL )
#if 0
  if ( (srb=calloc(sizeof(struct _srbinquiry), 1)) == ( char *) 0 )
#endif
    return(E$NoMem);

  srb->h.cmd = 0;           /* Host Adapter Inquiry */
  srb->h.adapter = inq->adapter_num;

  /* issue the request */
  if ( (i=aspireq(srb)) )  {             /* if error status */
    free(srb);
    return(i);
  }

  inq->adapters = srb->adapters;   /* number of hostadapters */
  inq->target_id = srb->target_id;

  for(i=0; i<16; ++i) inq->manager_id[i] = srb->manager_id[i];
  inq->manager_id[16] = '\0';

  for(i=0; i<16; ++i) inq->adapter_id[i] = srb->adapter_id[i];
  inq->adapter_id[16] = '\0';

  free(srb);
  return(E$NoErr);
}

/*
 * GetDeviceTyp: The returned typ corresponds to the typ of an
 *               SCSI-Inquiry command.
 */

int GetDeviceTyp(adapter, target_id, lun)
int adapter;
int target_id;
int lun;
{
  struct _srbgettyp *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) */

  if ( (srb=calloc(sizeof( struct _srbgettyp), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 1;
  srb->h.adapter = adapter;
  srb->target_id = target_id;
  srb->lun = lun;

  /* Issue the request */
  if ( (i=aspireq(srb)) ) {
    switch ( srb->h.status )  {
    case 0x82:
      i = E$NoDevice;
      break;
    case 0x80:
      i = E$AspiInval;
      break;
    case 0x81:
      i = E$NoAdapter;
      break;
    case 0x02:
      i = E$Abort;
      break;
    }
  } else i = srb->typ;

  ( void ) free(srb);
  return(i);
}


/*
 * The ScsiIoError function is used to generate an unique errorcode
 * from the three different errorcode sources.
 * The errorcode sources are:
 *   - The exit status of the aspi request
 *   - The host adapter status
 *   - The scsi target status
 *   - The sense data structure
 *
 * Ahhmm ..... actually this makes FOUR different sources not three :-)
 * The function is local to this module and called by all the SCSI-IO
 * functions upon exit.
 */

static int ScsiIoError(aspi_status, adapter_status, target_status, sense)
int aspi_status;
int adapter_status;
int target_status;
unsigned char *sense;
{
  int i;

  switch ( aspi_status )  {   /* check the aspi status */
  case 0x82:
    i = E$NoDevice;
    break;
  case 0x80:
    i = E$AspiInval;
    break;
  case 0x81:
    i = E$NoAdapter;
    break;
  case 0x02:
    i = E$Abort;
    break;
  case 0x04:
    /*
     * Ok, this means scsi-io-error, so we see if
     * we can find more details about the error.
     */
    i = E$IOErr;
    if ( adapter_status )  {
      switch ( adapter_status )  {
      case 0x11:
	i = E$SelTimeout;
	break;
      case 0x12:
	i = E$DataOverrun;
	break;
      case 0x13:
	i = E$BusFree;
	break;
      case 0x14:
	i = E$BusFail;
	break;
      }
    } else switch ( target_status )  {
        /*
         * If the adapter didn't show any errors
         * we going to check the target to see if
         * the target was the source of the error.
         */
    case 0x08:                         
      i = E$TargetBusy;
      break;
    case 0x18:
      i = E$Reservation;
      break;
    case 0x02:
        /* 
         * Allright, the target has send
         * sense data to the sense-data allocation
         * area at the end of the srb.
	 */
      if ( sense != ( unsigned char *) 0 ) switch ( sense[2] & 0x0f )  {
      case 0x00:  /* This means "no sense", but we check for End of Medium */
        if ( sense[2] & 0x40 ) i = E$EndOfMedium;
        break;
      case 0x01:
        i = E$NoErr;/* this would be pretty stupid to report, but anyway */
	break;
      case 0x02:
	i = E$NotReady;
	break;
      case 0x03:
	i = E$Medium;
	break;
      case 0x04:
	i = E$Hardware;
	break;
      case 0x05:
	i = E$IllegalReq;
	break;
      case 0x06:
        i = E$UnitAttention;
	break;
      case 0x07:
        i = E$DataProtect;
	break;
      case 0x08:
	i = E$BlankCheck;
	break;
      case 0x0b:
	i = E$TargetAbort;
	break;
      case 0x0d:
	i = E$VolOverflow;
	break;
      }
      break;
    }
    break;
  }
  return(i);
}

int SCSIInquiry(adapter, target_id, lun, inq, sense)
int adapter;
int target_id;
int lun;
scsiinquiry far *inq;
unsigned char *sense;
{
  struct _srbio *srb;
  int i,j;

  /* allocate a SCSI Request Block (SRB) */
  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 2;
  srb->h.adapter = adapter;
  srb->h.flags = (1<<3);
  srb->target_id = target_id;
  srb->lun = lun;
  srb->buf_seg = FP_SEG(inq);
  srb->buf_off = FP_OFF(inq);
  srb->alloc_len = sizeof( scsiinquiry );
  srb->sense_len = 14;
  srb->cdb_len = 6;
  srb->ccb.inq.cmd = 0x12;
  srb->ccb.inq.lun = lun;
  srb->ccb.inq.len = sizeof( scsiinquiry );

  if ( (i=aspireq(srb)) ) {
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.inq.sense);
  }

  if ( sense ) for(j=0; j<14; ++j) sense[j] = srb->ccb.inq.sense[j];

  ( void ) free(srb);
  return(i);
}


int SCSIResetDevice(adapter, target_id, lun)
int adapter;
int target_id;
int lun;
{
  struct _srbreset *srb;
  int i;


  /* allocate a SCSI Request Block (SRB) */

  if ( (srb=calloc(sizeof(struct _srbreset), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 4;
  srb->h.adapter = 0;
  srb->target_id = target_id;
  srb->lun = lun;

  if ( (i=aspireq(srb)) )
    i = ScsiIoError(srb->h.status, srb->adapter_status,
		    srb->target_status, ( unsigned char far * ) 0);

  ( void ) free (srb);
  return(i);
}

/*
 * The following commands are "Sequencial-Access Device" specific.
 */

int SCSIErase(adapter, target_id, lun, immediate, lng, sense)
int adapter;
int target_id;
int lun;
int immediate;
int lng;
unsigned char *sense;
{
  struct _srbio *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) */
  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 2;
  srb->h.adapter = adapter;
  srb->target_id = target_id;
  srb->lun = lun;
  srb->sense_len = 14;
  srb->cdb_len = 6;
  srb->ccb.c6.cmd = 0x19;
  srb->ccb.c6.flag0 = lng;
  srb->ccb.c6.flag1 = immediate;
  srb->ccb.c6.lun = lun;

  if ( (i=aspireq(srb)) ) {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.c6.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.c6.sense);
  }

  ( void ) free(srb);

  return(i);
}


int SCSILoad(adapter, target_id, lun, immediate, load, retention, eot, sense)
int adapter;
int target_id;
int lun;
int immediate;
int load;
int retention;
int eot;
unsigned char *sense;
{
  struct _srbio *srb;
  int i;


  /* allocate a SCSI Request Block (SRB) */
  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 2;
  srb->h.adapter = adapter;
  srb->target_id = target_id;
  srb->lun = lun;
  srb->sense_len = 14;
  srb->cdb_len = 6;
  srb->ccb.ld.cmd = 0x1b;
  srb->ccb.ld.immediate = immediate;
  srb->ccb.ld.lun = lun;
  srb->ccb.ld.load = load;
  srb->ccb.ld.retention = retention;
  srb->ccb.ld.eot = eot;

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.ld.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.ld.sense);
  }

  ( void ) free(srb);

  return(i);
}


#ifndef CTCTRL

int
SCSIRead(
    int         adapter,
    int         target_id,
    int         lun,
    char far    *buf,
    DWORD       xferlen,
    int         sili,
    int         fixed,
    unsigned char *sense
    )
{
  struct _srbio *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) in the lower 1 MB */
  if ( (srb=calloc(sizeof(struct _srbio),1)) == 0 ) {
        return(E$NoMem);
  }

  srb->h.cmd = 2;           /* scsi I/O request */
  srb->h.adapter = (unsigned char)adapter;
  srb->h.flags = (1<<3);
  srb->target_id = (unsigned char)target_id;
  srb->lun = (unsigned char)lun;
  srb->alloc_len = fixed ? (xferlen * physRecSize) : xferlen;
  srb->sense_len = 14;
  srb->buf_seg = FP_SEG(buf);
  srb->buf_off = FP_OFF(buf);
  srb->cdb_len = 6;
  srb->ccb.rdwr.cmd = 0x08;
  srb->ccb.rdwr.fixed = fixed;
  srb->ccb.rdwr.sili = sili;
  srb->ccb.rdwr.lun = lun;
  srb->ccb.rdwr.len_0 = (unsigned)(xferlen & 0xff);
  srb->ccb.rdwr.len_1 = (unsigned)((xferlen>>8) & 0xff);
  srb->ccb.rdwr.len_2 = (unsigned)((xferlen>>16) & 0xff);

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.rdwr.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.rdwr.sense);
  }

  free(srb);
  return(i);
}
#endif  /* CTCTRL */

/*
 * Issue a Mode Sense CDB to get the Parameter List Header and
 *  Block descriptor.  Convert the values to internal format.
 *  R. Rose - Feb '93
 */
int
SCSIModeSense(
    int         adapter,
    int         target_id,
    int         lun,
    modeBlockSz *mode,
    unsigned char   *sense
    )
{
    struct _srbio       *srb;
    unsigned char far   *modeParm;
    int                 i,rc;

    /* allocate an ASPI SRB and a data buffer */
    if ( 0 == (srb=calloc(sizeof(struct _srbio), 1)) ) {
        return(E$NoMem);
    }
    if ( 0 == (modeParm=malloc(12)) ) {
        free(srb);
        return(E$NoMem);
    }

    /* Fill in the SRB for the ASPI request */
    srb->h.cmd = 2;
    srb->h.adapter = (unsigned char)adapter;
    srb->h.flags = 0x8;
    srb->target_id = (unsigned char)target_id;
    srb->lun = (unsigned char)lun;
    srb->alloc_len = 12;
    srb->sense_len = 14;
    srb->buf_seg = FP_SEG(modeParm);
    srb->buf_off = FP_OFF(modeParm);
    srb->cdb_len = 6;
    srb->ccb.c6.cmd = 0x1A;
    srb->ccb.c6.lun = lun;
    srb->ccb.c6.len = 0xC;

    /*
     * Issue ASPI request, await completion and convert data to
     *  internal format.
     */
    if ( (aspireq(srb)) )  {
        if (sense) {
            for (i=0; i<14; ++i) {
                sense[i] = srb->ccb.c6.sense[i];
            }
        }
        rc = ScsiIoError(srb->h.status, srb->adapter_status,
                         srb->target_status, srb->ccb.c6.sense);
    } else  {
        mode->header.dataLen = modeParm[0];
        mode->header.mediumType = modeParm[1];
        mode->header.devSpecific = modeParm[2];
        mode->header.bdescLen = modeParm[3];

        mode->bdesc.denCode = modeParm[4];
        mode->bdesc.nBlocks = ((DWORD)modeParm[5]<<16) |
                              ((DWORD)modeParm[6]<<8) |
                              ((DWORD)modeParm[7]);
        mode->bdesc.blockLen = ((DWORD)modeParm[9]<<16) |
                               ((DWORD)modeParm[10]<<8) |
                               ((DWORD)modeParm[11]);
        rc = E$NoErr;
    }

    free((void *)modeParm);
    free(srb);
    return(rc);
}

/*
 * Issue a Mode Select CDB to set the Buffer-mode, speed, recording
 *  density and blocksize.  Convert the values from internal format.
 *  R. Rose - Feb '93
 */
int
SCSIModeSelect(
    int         adapter,
    int         target_id,
    int         lun,
    char        modePf,
    modeBlockSz *mode,
    unsigned char   *sense
    )
{
    struct _srbio       *srb;
    unsigned char far   *modeParm;
    int                 i,rc;

    /* allocate an ASPI SRB and a data buffer */
    if ( 0 == (srb=calloc(sizeof(struct _srbio), 1)) ) {
        return(E$NoMem);
    }
    if ( 0 == (modeParm=calloc(12,1)) ) {
        free(srb);
        return(E$NoMem);
    }

    /* Fill in the SRB for the ASPI request */
    srb->h.cmd = 2;
    srb->h.adapter = (unsigned char)adapter;
    srb->h.flags = 0x10;
    srb->target_id = (unsigned char)target_id;
    srb->lun = (unsigned char)lun;
    srb->alloc_len = 12;
    srb->sense_len = 14;
    srb->buf_seg = FP_SEG(modeParm);
    srb->buf_off = FP_OFF(modeParm);
    srb->cdb_len = 6;
    srb->ccb.c6.cmd = 0x15;
    srb->ccb.c6.lun = lun;
    srb->ccb.c6.flag4 = modePf ? 0x1 : 0x0;
    srb->ccb.c6.len = 0xC;

    /*
     * Convert mode parameters to external format and issue
     *  ASPI request.
     */
    modeParm[0] = mode->header.dataLen;
    modeParm[1] = mode->header.mediumType;
    modeParm[2] = mode->header.devSpecific;
    modeParm[3] = mode->header.bdescLen;

    modeParm[4] = mode->bdesc.denCode;
    modeParm[5] = (unsigned char)((mode->bdesc.nBlocks>>16) & 0xFF);
    modeParm[6] = (unsigned char)((mode->bdesc.nBlocks>>8) & 0xFF);
    modeParm[7] = (unsigned char)(mode->bdesc.nBlocks & 0xFF);

    modeParm[9]  = (unsigned char)((mode->bdesc.blockLen>>16) & 0xFF);
    modeParm[10] = (unsigned char)((mode->bdesc.blockLen>>8) & 0xFF);
    modeParm[11] = (unsigned char)(mode->bdesc.blockLen & 0xFF);

    if ( (aspireq(srb)) )  {
        if (sense) {
            for (i=0; i<14; ++i) {
                sense[i] = srb->ccb.c6.sense[i];
            }
        }
        rc = ScsiIoError(srb->h.status, srb->adapter_status,
                         srb->target_status, srb->ccb.c6.sense);
    } else  {
        rc = E$NoErr;
    }

    free((void *)modeParm);
    free(srb);
    return(rc);
}


int SCSIReadBlockLimits(adapter, target_id, lun, lim, sense)
int adapter;
int target_id;
int lun;
blocklimit *lim;
unsigned char *sense;
{
  struct _srbio *srb;
  char *buf;
  unsigned char far *block;
  int i;

  /* allocate a SCSI Request Block (SRB) */
  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  /* allocate small data buffer */
  if ( (buf=calloc(64, 1)) ==  0 ) return(E$NoMem);

  block = ( unsigned char far * ) buf;

  srb->h.cmd = 2;
  srb->h.adapter = adapter;
  srb->h.flags = (1<<3);
  srb->target_id = target_id;
  srb->lun = lun;
  srb->alloc_len = 6;
  srb->sense_len = 14;
  srb->buf_seg = FP_SEG(block);
  srb->buf_off = FP_OFF(block);
  srb->cdb_len = 6;
  srb->ccb.c6.cmd = 0x05;
  srb->ccb.c6.lun = lun;

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.c6.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.c6.sense);
  } else  {
    lim->max =   (    ((( DWORD ) buf[1])<<16) |
                      ((( DWORD ) buf[2])<<8)  |
                      ((( DWORD ) buf[3])) );
    lim->min =   (    ((( DWORD ) buf[4])<<8) |
                      ((( DWORD ) buf[5])) );
  }

  ( void ) free(buf);
  ( void ) free(srb);

  return(i);
}

int SCSIRewind(adapter, target_id, lun, immediate, sense)
int adapter;
int target_id;
int lun;
int immediate;
unsigned char *sense;
{
  struct _srbio *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) */
  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 2;
  srb->h.adapter = adapter;
  srb->target_id = target_id;
  srb->lun = lun;
  srb->sense_len = 14;
  srb->cdb_len = 6;
  srb->ccb.c6.cmd = 0x01;
  srb->ccb.c6.lun = lun;
  srb->ccb.c6.flag0 = immediate;

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.c6.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.c6.sense);
  }

  ( void ) free(srb);

  return(i);
}


int SCSISpace(adapter, target_id, lun, code, count, sense)
int adapter;
int target_id;
int lun;
int code;
long count;
unsigned char *sense;
{
  DWORD lcount;
  struct _srbio *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) */
  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  lcount = count;

  srb->h.cmd = 2;
  srb->h.adapter = adapter;
  srb->target_id = target_id;
  srb->lun = lun;
  srb->sense_len = 14;
  srb->cdb_len = 6;
  srb->ccb.spc.cmd = 0x11;
  srb->ccb.spc.code = code;
  srb->ccb.spc.lun = lun;
  srb->ccb.spc.cnt0 = (lcount & 0xff);
  srb->ccb.spc.cnt1 = ((lcount>>8) & 0xff);
  srb->ccb.spc.cnt2 = ((lcount>>16) & 0xff);

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.spc.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.spc.sense);
  }

  ( void ) free(srb);

  return(i);
}


#ifndef CTCTRL
int
SCSIWrite(
    int     adapter,
    int     target_id,
    int     lun,
    char far *buf,
    DWORD   xferlen,
    int     fixed,
    unsigned char *sense
    )
{
  struct _srbio *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) in the lower 1 MB */

  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  srb->h.cmd = 2;           /* scsi I/O request */
  srb->h.adapter = (unsigned char)adapter;
  srb->h.flags = (1<<4);
  srb->target_id = (unsigned char)target_id;
  srb->lun = (unsigned char)lun;
  srb->alloc_len = fixed ? (xferlen * physRecSize) : xferlen;
  srb->sense_len = 14;
  srb->buf_seg = FP_SEG(buf);
  srb->buf_off = FP_OFF(buf);
  srb->cdb_len = 6;
  srb->ccb.rdwr.cmd = 0x0a;
  srb->ccb.rdwr.fixed = fixed;
  srb->ccb.rdwr.lun = lun;
  srb->ccb.rdwr.len_0 = (unsigned)(xferlen & 0xff);
  srb->ccb.rdwr.len_1 = (unsigned)((xferlen>>8) & 0xff);
  srb->ccb.rdwr.len_2 = (unsigned)((xferlen>>16) & 0xff);

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.rdwr.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.rdwr.sense);
  }

  free(srb);
  return(i);
}
#endif  /* CTCTRL */


int SCSIWriteFilemarks(adapter, target_id, lun, len, setmark, immediate, sense)
int adapter;
int target_id;
int lun;
DWORD len;
int setmark;
int immediate;
unsigned char *sense;
{
  DWORD length;
  struct _srbio *srb;
  int i;

  /* allocate a SCSI Request Block (SRB) */

  if ( (srb=calloc(sizeof(struct _srbio), 1)) == 0 ) return(E$NoMem);

  length = len;

  srb->h.cmd = 2;           /* scsi I/O request */
  srb->h.adapter = adapter;
  srb->target_id = target_id;
  srb->lun = lun;
  srb->sense_len = 14;
  srb->cdb_len = 6;
  srb->ccb.rdwr.cmd = 0x10;
  srb->ccb.rdwr.fixed = immediate;
  srb->ccb.rdwr.sili = setmark;
  srb->ccb.rdwr.lun = lun;
  srb->ccb.rdwr.len_0 = (length & 0xff);
  srb->ccb.rdwr.len_1 = ((length>>8) & 0xff);
  srb->ccb.rdwr.len_2 = ((length>>16) & 0xff);

  if ( (i=aspireq(srb)) )  {
    if ( sense ) for(i=0; i<14; ++i) sense[i] = srb->ccb.rdwr.sense[i];
    i = ScsiIoError(srb->h.status, srb->adapter_status,
                    srb->target_status, srb->ccb.rdwr.sense);
  }

  ( void ) free(srb);
  return(i);
}

#ifdef TEST_ASPI
main()
{
  aspiinquiry inq;
  scsiinquiry sinq;
  int i, j;
  long l;
  char *buf;

  aspiinit();

  inq.adapter_num = 0;
  HostAdapterInquiry(&inq);


  printf("%d adapters, target:%d\n", inq.adapters, inq.target_id);
  printf("Manager id: %s\n", inq.manager_id);
  printf("adapter id: %s\n", inq.adapter_id);

  printf("\nrecsize: %ld, log2(recsize)= %d\n", recsize, logrec);

  if ( (i=SCSIInquiry(TARGET, &sinq, 0)) != E$NoErr )  {
    printf("Scsi Inquiry returned %d\n", i);
  } else  {
    printf("type,pqual,dqual: %d, %d, %d\n", sinq.per_type,
            sinq.per_qualify, sinq.dev_qualify);
    printf("vendor_id:%.8s\n", sinq.vendor_id);
    printf("product_id:%.16s\n", sinq.product_id);
    printf("revision:%.4s\n", sinq.revision);
    printf("vendor:%.20s\n", sinq.vendor);
  }

  if ( (buf=malloc(32768)) == 0 ) {
    printf("can't alloc buf\n");
    exit(0);
  }

  printf("\nBlocklimits, max:%ld min:%ld\n", lim.max, lim.min);

  i = SCSILoad(TARGET, 0, 1, 0, 0, 0);
  printf("\nLoad returned %d\n -->", i);

#if 0
  printf("seeking to tape file 2...\n");

  if ( (i=SCSISpace(TARGET, 1, ( DWORD ) 1, 0)) )
    printf("Error %d\n", i );
  else printf("OK\n");

  printf("seeking to tape block 400 on file 2...\n");
  if ( (i=SCSISpace(TARGET, 0, ( DWORD ) 400, 0)) )
    printf("Error %d\n", i );
  else printf("OK\n");

  printf("Writing two sectors..(this should return an error)...\n");
  i = SCSIWrite(TARGET, buf, 1024l, 9, 1, 0);
  printf("Write returned %d\n", i);

  printf("seeking back two sectors...\n");
  if ( (i=SCSISpace(TARGET, 0, -2l, 0)) )
    printf("Error %d\n", i );
  else printf("OK\n");

  printf("reading back the two sectors...\n");
  i = SCSIRead(TARGET, buf, 1024l, 9, 0, 1, 0);
  printf("Read returned %d\n", i);

  printf("rewinding the tape...\n");
  i = SCSIRewind(TARGET, 0, 0);
  printf("Rewind returned %d\n", i);

  printf("seeking to the end of the data...\n");
  if ( (i=SCSISpace(TARGET, 3, 0l, 0)) )
    printf("Error %d\n", i );
  else printf("OK\n");

  printf("Writing two sectors...\n");
  i = SCSIWrite(TARGET, buf, 1024l, 9, 1, 0);
  printf("Write returned %d\n", i);

  printf("Writing two sectors...\n");
  i = SCSIWrite(TARGET, buf, 1024l, 9, 1, 0);
  printf("Write returned %d\n", i);

  printf("Writing a file mark !\n");
  i = SCSIWriteFilemarks(TARGET, 1l, 0, 0, 0);
  printf("Write returned %d\n", i);

  printf("rewinding the tape...\n");
  i = SCSIRewind(TARGET, 0, 0);
  printf("Rewind returned %d\n", i);

  printf("Now counting files ...\n");
  i = 0;
  while ( !SCSISpace(TARGET, 1, 1l, 0) ) ++ i;
  printf("... there are %d files on the tape now\n",i);

  printf("rewinding the tape...\n");
  i = SCSIRewind(TARGET, 0, 0);
  printf("Rewind returned %d\n", i);

  printf("eraseing the tape...\n");
  i = SCSIErase(TARGET, 0, 0, 0);
  printf("erase returned %d\n", i);
#endif

  l = 0l;

  printf("Now see what kind of errors we get at Overflow\n");
  while ( l < 4080l ) {
    i = SCSIRead(TARGET, buf, 32768l, 9, 0, 1, rsense);
    if ( i ) {
      printf("reading block # %ld: errno: %d rsense:", l, i);
      for(j=0; j<14; ++j) printf(" %02x", rsense[j]);
      printf("\n");
    }
    ++l;
  }

/*
  while ( !(i=SCSIWrite(TARGET, buf, 32768l, 9, 1, rsense)) )  ++l;
  printf("last block was %ld\n", l);
  printf("errno: %d sense:", i);
  for(j=0; j<14; ++j) printf(" %02x", rsense[j]);
  printf("\n");
*/

  i = SCSILoad(TARGET, 0, 0, 0, 0, 0);
  printf("Unload returned %d\n -->", i);
  free(buf);
}
#endif
